{
 *************************************************************************
 *
 *                           PEUNIT.PAS
 *
 * A Pascal UNIT for parsing Win32 PE header information
 *
 * Author : Prof. Abimbola A Olowofoyeku <African_Chief@bigfoot.com>
 *
 * Supports : virtually all Pascal compilers;
 *   * GNU Pascal, Turbo Pascal, Borland Pascal, Virtual Pascal
 *   * Delphi
 *
 * Licence: This code is released as FREEWARE, and is supplied
 *          WITHOUT ANY WARRANTIES WHATSOEVER
 *
 * (c)2000,2001 Prof. Abimbola A Olowofoyeku (The African Chief)
 *
 **************************************************************************
}
UNIT PEUnit;

INTERFACE
{$I-}

{$ifdef __GPC__}
TYPE
Word    = Cardinal ( 16 );
Longint = Integer;
{$endif}

TYPE
  IMAGE_DOS_HEADER = RECORD     { DOS 1, 2, 3 .EXE header     }
    e_magic   : Word;     { Magic number                      }
    e_cblp    : Word;     { Words on last page of file        }
    e_cp      : Word;     { Pages in file                     }
    e_crlc    : Word;     { Relocations                       }
    e_cparhdr : Word;     { Size of header in paragraphs      }
    e_minalloc : Word;    { Minimum extra paragraphs needed   }
    e_maxalloc : Word;    { Maximum extra paragraphs needed   }
    e_ss      : Word;     { Initial (relative) SS value       }
    e_sp      : Word;     { Initial SP value                  }
    e_csum    : Word;     { Checksum                          }
    e_ip      : Word;     { Initial IP value                  }
    e_cs      : Word;     { Initial (relative) CS value       }
    e_lfarlc  : Word;     { File address of relocation table  }
    e_ovno    : Word;     { Overlay number                    }
    e_res     : ARRAY [0..3] OF Word;  { Reserved words        }
    e_oemid   : Word;     { OEM identifier (for e_oeminfo)    }
    e_oeminfo : Word;     { OEM information; e_oemid specific }
    e_res2    : ARRAY [0..9] OF Word;  { Reserved words        }
    e_lfanew  : Longint;  { File address of new exe header    }
    END;

{ signatures }
CONST
  IMAGE_DOS_SIGNATURE    = $00005A4D; { MZ    }
  IMAGE_OS2_SIGNATURE    = $0000454E; { NE    }
  IMAGE_OS2_SIGNATURE_LE = $00005A4D; { LE    }
  IMAGE_NT_SIGNATURE     = $00004550; { PE00  }

{ Win32 PE header }
TYPE
IMAGE_PE_HEADER = RECORD
     Machine : Word;
     NumberOfSections : Word;
     TimeDateStamp : Longint;
     PointerToSymbolTable : Longint;
     NumberOfSymbols : Longint;
     SizeOfOptionalHeader : Word;
     Characteristics : Word;
END; { size = 20 bytes }

{ Win32 PE data directory }
TYPE
DATA_DIRECTORY = RECORD
     VirtualAddress : Longint;
     Size : Longint;
END; { size = 8 bytes }

{  array of data directories }
TYPE
DataDirectoryArray = ARRAY [0..15] OF Data_Directory; { size = 128 bytes }

{ Win32 PE "optional" header }
TYPE
IMAGE_OPTIONAL_HEADER = RECORD
{     //
     // Standard fields.
     //}
     Magic : Word;                 {2}

     MajorLinkerVersion,
     MinorLinkerVersion : Byte;    {4}

     SizeOfCode,                   {8}
     SizeOfInitializedData,        {12}
     SizeOfUninitializedData,      {16}
     AddressOfEntryPoint,          {20}
     BaseOfCode,                   {24}
     BaseOfData : Longint;         {28}
     {
     //
     // NT additional fields.
     //}
     ImageBase,                    {32}
     SectionAlignment,             {36}
     FileAlignment : Longint;      {40}

     MajorOperatingSystemVersion,  {42}
     MinorOperatingSystemVersion,  {44}
     MajorImageVersion,            {46}
     MinorImageVersion,            {48}
     MajorSubsystemVersion,        {50:offset=48}
     MinorSubsystemVersion : Word; {52:offset=50}

     Reserved1,
     SizeOfImage,
     SizeOfHeaders,
     CheckSum : Longint;

     Subsystem,
     DllCharacteristics : Word;

     SizeOfStackReserve,
     SizeOfStackCommit,
     SizeOfHeapReserve,
     SizeOfHeapCommit,
     LoaderFlags,
     NumberOfRvaAndSizes : Longint;
     DataDirectory : DataDirectoryArray;
 END;

{ section headers }
TYPE MiscRec = RECORD
    CASE Integer OF
      0 : ( PhysicalAddress : Longint );
      1 : ( VirtualSize : Longint );
END;

TYPE IMAGE_SECTION_HEADER = RECORD
    Name : ARRAY [0..7] OF Char;
    Reserved : MiscRec;
    VirtualAddress,
    SizeOfRawData,
    PointerToRawData,
    PointerToRelocations,
    PointerToLinenumbers : Longint;
    NumberOfRelocations,
    NumberOfLinenumbers  : Word;
    Characteristics : Longint;
END;  { size = 40 bytes }

{ exports headers }
TYPE IMAGE_EXPORT_DIRECTORY  = RECORD
    Characteristics,
    TimeDateStamp : Longint;

    MajorVersion,
    MinorVersion : Word;

    Name,
    Base,
    NumberOfFunctions,
    NumberOfNames : Longint;

    AddressOfFunctions,
    AddressOfNames : ^Longint;
    AddressOfNameOrdinals : ^Word;
END;  { size = 40 bytes }

{ imports headers }
TYPE IMAGE_IMPORT_MODULE_DIRECTORY = RECORD
    dwRVAFunctionNameList,
    dwUseless1,
    dwUseless2,
    dwRVAModuleName,
    dwRVAFunctionAddressList : Longint;
END; { Size = 20 bytes }

{ structure for reading the PE whole header in one go }
TYPE
pPEHeaderr = ^PE_HeaderRec;
PE_HeaderRec = RECORD
   DosHeader : IMAGE_DOS_HEADER;
   PEHeader  : IMAGE_PE_HEADER;
   OptionalHeader : IMAGE_OPTIONAL_HEADER;
   SectionHeader : IMAGE_SECTION_HEADER;
END;

{ exported functions }

{  return the offset to the PE header in an open PE binary }
FUNCTION PEHeaderOffSet ( VAR f : File ) : Longint;

{  return the offset to the PE signature in an open PE binary }
FUNCTION PESignatureOffSet ( VAR f : File ) : Longint;

{  return the offset to the optional PE header in an open PE binary }
FUNCTION PEOptionalHeaderOffSet ( VAR f : File ) : Longint;

{  return the offset to the PE section header in an open PE binary }
FUNCTION PESectionHeaderOffSet ( VAR f : File ) : Longint;


{  read the PE whole header information }
FUNCTION ReadWholePEHeader ( VAR f : File; VAR h : PE_HeaderRec ) : Longint;

{  read only the PE image header }
FUNCTION ReadPEHeader ( VAR f : File; VAR a : IMAGE_PE_HEADER ) : Longint;

{  read only the PE optional header }
FUNCTION ReadOptionalHeader ( VAR f : File; VAR a : IMAGE_OPTIONAL_HEADER ) : Longint;

{  Write only the PE optional header }
FUNCTION WriteOptionalHeader ( VAR f : File; VAR a : IMAGE_OPTIONAL_HEADER ) : Longint;

{  return the offset to the entry point }
FUNCTION PEEntryPoint ( VAR f : File ) : Longint;

{  return the offset to the section header }
FUNCTION SectionHeaderOffSet ( VAR f : File; SectionName : pChar;
                             VAR Return : IMAGE_SECTION_HEADER ) : Longint;
{  return the number of sections }
FUNCTION PESectionCount ( VAR f : File ) : Longint;


IMPLEMENTATION

USES {$ifdef Win32}SysUtils{$else}Strings{$endif};

TYPE
{$ifdef __GPC__}
 myInt   = Integer;
{$else}
 {$ifdef Win32}
 MyInt = Longint;
 {$else}
 MyInt = Word;
 {$endif}
{$endif}

FUNCTION PESignatureOffSet ( VAR f : File ) : Longint;
VAR
i : Longint;
a : IMAGE_DOS_HEADER;
w : MyInt;
BEGIN
  PESignatureOffSet := - 300;
  i := FilePos ( f );
  Seek ( f, 0 );
  BlockRead ( f, a, sizeof ( a ), w );
  Seek ( f, i );
  IF w = sizeof ( a ) THEN PESignatureOffSet := a.e_lfanew;
END;

FUNCTION PEHeaderOffSet ( VAR f : File ) : Longint;
BEGIN
  PEHeaderOffSet := PESignatureOffSet ( f ) + 4;
END;

FUNCTION PEOptionalHeaderOffSet ( VAR f : File ) : Longint;
BEGIN
  PEOptionalHeaderOffSet := PEHeaderOffSet ( f ) + Sizeof ( IMAGE_PE_HEADER );
END;

FUNCTION PESectionHeaderOffSet ( VAR f : File ) : Longint;
BEGIN
  PESectionHeaderOffSet := PEOptionalHeaderOffSet ( f ) + Sizeof ( IMAGE_OPTIONAL_HEADER );
END;

FUNCTION ReadPEHeader ( VAR f : File; VAR a : IMAGE_PE_HEADER ) : Longint;
VAR
i, j : Longint;
w    : MyInt;
BEGIN
   ReadPEHeader := - 300;
   i := FilePos ( f );
   j := PEHeaderOffSet ( f );
   Seek ( f, j );
   BlockRead ( f, a, sizeof ( a ), w );
   IF w = sizeof ( a ) THEN ReadPEHeader := FilePos ( f );
   Seek ( f, i );
END;

FUNCTION ReadOptionalHeader ( VAR f : File; VAR a : IMAGE_OPTIONAL_HEADER ) : Longint;
VAR
i, j : Longint;
w : MyInt;
BEGIN
  ReadOptionalHeader := - 300;
  i := FilePos ( f );
  j := PEOptionalHeaderOffSet ( f );
  Seek ( f, j );
  BlockRead ( f, a, sizeof ( a ), w );
  { return end of optional header: same as beginning of section headers }
  IF w = sizeof ( a ) THEN ReadOptionalHeader := FilePos ( f );
  Seek ( f, i );
END;

FUNCTION WriteOptionalHeader ( VAR f : File; VAR a : IMAGE_OPTIONAL_HEADER ) : Longint;
VAR
i, j : Longint;
w : MyInt;
BEGIN
  WriteOptionalHeader := - 300;
  i := FilePos ( f );
  j := PEOptionalHeaderOffSet ( f );
  Seek ( f, j );
  BlockWrite ( f, a, sizeof ( a ), w );

  { return end of optional header: same as beginning of section headers }
  IF w = sizeof ( a ) THEN WriteOptionalHeader := FilePos ( f );
  Seek ( f, i );
END;

FUNCTION PEEntryPoint ( VAR f : File ) : Longint;
VAR
a : IMAGE_OPTIONAL_HEADER;
BEGIN
  PEEntryPoint := - 300;
  IF ReadOptionalHeader ( f, a ) > 0 THEN PEEntryPoint := a.AddressOfEntryPoint;;
END;

FUNCTION PESectionCount ( VAR f : File ) : Longint;
VAR
a : IMAGE_PE_HEADER;
BEGIN
  PESectionCount := 0;
  IF ReadPEHeader ( f, a ) > 0 THEN PESectionCount := a.NumberOfSections;
END;

FUNCTION ReadWholePEHeader ( VAR f : File; VAR h : PE_HeaderRec ) : Longint;
VAR
i    : Longint;
w    : MyInt;
s    : Word;
LABEL FallOut;

BEGIN
   i := FilePos ( f );
   Seek ( f, 0 );
   WITH h DO BEGIN
       ReadWholePEHeader := - 1; { not an executable }
       BlockRead ( f, DosHeader, sizeof ( DosHeader ), w );
       WITH DosHeader
       DO BEGIN
          IF ( w <> sizeof ( DosHeader ) )
          OR ( e_magic <> IMAGE_DOS_SIGNATURE )  THEN GOTO FallOut;

          ReadWholePEHeader := - 2; { dos program }
          IF ( FileSize ( f ) = longint ( pred ( e_cp ) ) * 512 + e_cblp )
          THEN GOTO FallOut;

          seek ( f, e_lfanew );
          blockread ( f, s, sizeof ( s ), w );

          ReadWholePEHeader := - 3; { Win16 or OS/2 program }
          { check that it is a PE program }
          CASE s OF
               IMAGE_OS2_SIGNATURE : GOTO FallOut;
               IMAGE_OS2_SIGNATURE_LE :
               BEGIN
                 ReadWholePEHeader := - 4; { OS/2 program }
                 GOTO FallOut;
               END;
          END;

          { read PE header }
          seek ( f, e_lfanew + 4 );
          blockread ( f, PEHeader, sizeof ( PEHeader ), w ); { pe header }

          { read PE optional header }
          seek ( f, e_lfanew + 24 );
          blockread ( f, OptionalHeader, sizeof ( OptionalHeader ), w ); { pe optional header }

          ReadWholePEHeader := FilePos ( f ); { return end of optional header }
          GOTO FallOut;
       END;
   END;

FallOut :
 BEGIN
   Seek ( f, i );
 END;
END;

FUNCTION SectionHeaderOffSet ( VAR f : File; SectionName : pChar;
                             VAR Return : IMAGE_SECTION_HEADER ) : Longint;
VAR
r, i, k, l : longint;
w : MyInt;
BEGIN
   SectionHeaderOffSet := - 1;
   { read sections }
   l := PESectionHeaderOffSet ( f );
   IF l <= 0 THEN Exit;

   SectionHeaderOffSet := - 2;
   k := PESectionCount ( f );
   IF k <= 0 THEN Exit;

   SectionHeaderOffSet := 0; { section not found }
   i := FilePos ( f );
   Seek ( f, l );
   FOR r := 0 TO Pred ( k )
   DO BEGIN
      BlockRead ( f, Return, Sizeof ( Return ), w );
      IF StriComp ( Return.Name, SectionName ) = 0
      THEN BEGIN
         { return offset to section header }
         SectionHeaderOffSet := FilePos ( f ) - SizeOf ( Return );
         Break;
      END;
   END;
   Seek ( f, i );
END;


END.

