{ **********************************************************************
  **    THIS CODE IS FREEWARE, AND IS SUPPLIED WITH NO WARRANTIES
  **
  **                  htz_unit.pas
  **
  **    a unit to decompress or create "htz" files
  **    uses the PasZlib libraries from:
  **            http://www.nomssi.de/paszlib/paszlib.html
  **
  **   Author:
  **       Prof Abimbola A Olowofoyeku (The African Chief)
  **            http://www.greatchief.plus.com/
  **
  **       Version:             1.04.9
  **       Last Modified:       5th August 2004
  **********************************************************************
}

UNIT htz_unit;

INTERFACE

{$I-,R-}

USES
zutil,
zinflate,
zdeflate,
zlib,
zcompres,
zuncompr,
sysutils;


CONST

{ program name }
ProgName = 'htz';

{ Author }
Author = 'Professor Abimbola A Olowofoyeku';

{ version number }
Version_Num  = '1.04.9';

{ date of last edit }
Version_Date = '5th August 2004';

{ htz files start with this signature }
htz_file_signature = #18#52#0#0;  { signature for htz files }
z_deflate_sig      = #8;          {  zdeflate marker }

{ file sharing }
FileModeShare = fmOpenRead OR fmShareDenyNone;

{ assorted type definitions }
TYPE
sig_type = PACKED ARRAY [0..3] OF char;

{ 16-bit Word }
htz_Word = ush;

{ header structure for htz files }
TYPE
htz_header = PACKED RECORD
  signature : ARRAY [1..3] OF byte; { 3 bytes   : should be $12 $34 $00 }
  Size_Main : htz_Word;             { 4-5 bytes : size 1 }
  Size_Rem  : Byte;                 { 6th byte  : size 2 }
  Last1     : Byte;                 { 7th byte  : $87 }
  Last2     : Byte;                 { 8th byte  : $65 }
  zlib_sig  : char;                 { 9th byte: should be #8 }
END;

{ callback function type }
TYPE
htz_CallBack = FUNCTION ( CONST Data; CONST DataSize : Longint ) : Longint STDCALL;
{ data = a buffer containing processed data
  datasize = the size of that data
  Return value: return >= 0 if you want processing to continue, or a negative value
                for processing to stop
}


{ *** high level routines *** }
FUNCTION CompressFile ( Src, Target : pChar ) : Longint; STDCALL;
{
  Compress a file - parameters are source file and target file
  returns negative value if there is an error, else the number of
  bytes written
}

FUNCTION CompressFileEx ( Src, Target : pChar; CallBack : htz_CallBack ) : Longint;
STDCALL;
{
  Compress a file - with callback functionality
}


FUNCTION UnCompressFile ( Src, Target : pChar ) : Longint; STDCALL;
{
  uncompress a file -
  parameters are source file and target file
  returns negative value if there is an error, else the number of
  bytes extracted
}

FUNCTION UnCompressFileEx ( Src, Target : pChar; CallBack : htz_CallBack  ) : Longint;
STDCALL;
{
  uncompress a file - with callback functionality
}

FUNCTION IsHTZFile ( FName : pChar ) : Longint; STDCALL;
{ rudimentary check for whether a file is an HTZ or other zlib file
  returns > 0 if yes, and <= 0 if not
}

FUNCTION HTZFileSize ( FName : pChar ) : Longint; STDCALL;
{returns the uncompressed size of the HTZ file (if it is one) or
 actual file size
}

FUNCTION htz_Version : pChar; STDCALL;
{ returns the version number
}

FUNCTION htz_Author : pChar; STDCALL;
{ returns the author's name
}

FUNCTION htz_Date : pChar; STDCALL;
{ returns the date of the library
}

IMPLEMENTATION

// USES Dialogs;
{ @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ }
FUNCTION ReadCh ( VAR f : FILE ) : char;
{ read a character in a file that is already opened, and move the
  file pointer by one
}
VAR
Ch : Char;
code : uInt;
BEGIN
   BlockRead ( f, Ch, Sizeof ( Ch ), Code );
   ReadCh := Ch;
END; { ReadCh }

FUNCTION FindDeflateStart ( VAR f : FILE ) : Longint;
{
  find the first #8 char in a file and stay there, or rewind to the
  beginning
 }
VAR
err : Integer;
BEGIN
  err := 0;
  Seek ( f, 0 );
  WHILE ( ReadCh ( f ) <> z_deflate_sig )
  DO BEGIN
    Inc ( err );
    IF err > 128 THEN Break;  { read only the first 128 bytes }
  END;
  IF err > 128 THEN Seek ( f, 0 );
  FindDeflateStart := FilePos ( f );
END; { FindDeflateStart }

FUNCTION IsHTZFile ( FName : pChar ) : Longint;
// STDCALL;
VAR
  fin : FILE;
  aResult : Longint;
  four : sig_type;
BEGIN
  Result := - 1;
  assign ( fin, Strpas ( FName ) );
  aResult := FileMode;
  FileMode := FileModeShare;
  reset ( fin, 1 );  { open }
  FileMode := aResult;
  IF IoResult <> 0 THEN Exit;

  Result := - 2;
  { read first 4 bytes for htz signature }
  BlockRead ( fin, four, sizeof  ( four ), aResult );
  IF ( strpas ( four ) = strpas ( htz_file_signature ) )
  THEN BEGIN
     Result := FindDeflateStart ( fin ) { = 9};
  END ELSE BEGIN
     aResult := FindDeflateStart ( fin ) { > 0};
     IF aResult > 0 THEN Result := aResult;
  END;
  Close ( fin );
END; { IsHTZFile }

FUNCTION GetOutBytes ( CONST InBytes : Longint ) : Longint;
{ return a much bigger buffer than the supplied one }
BEGIN
  GetOutBytes := InBytes * 6;
END; { GetoutBytes }

FUNCTION UnCompressFileEx ( Src, Target : pChar;
CallBack : htz_CallBack ) : Longint;
// STDCALL;
VAR
  err : int;
  z : z_stream; { decompression stream }
  input_buffer,
  output_buffer : pzByteArray;
  t1, t2 : Longint;
  in_count : uLong;
  fin, fout : FILE;
  aResult : Longint;
  uSize : Cardinal;
  four : sig_type;
  HeadSize : PACKED RECORD
     Size : htz_Word;
     Remainder : Byte;
  END;

BEGIN
  UnCompressFileEx := - 1;

  { initialize input file }
  err := FileMode;
  FileMode := FileModeShare;
  assign ( fin, Strpas ( Src ) );
  reset ( fin, 1 );  { open }
  FileMode := err;
  IF IoResult <> 0 THEN Exit;

  { initialize output file }
  UnCompressFileEx := - 2;
  assign ( fout, Strpas ( Target ) );
  rewrite ( fout, 1 ); { create }
  IF IoResult <> 0
  THEN BEGIN
     Close ( fin );
     Exit;
  END;

  { initialize z_stream }
  z.zalloc := NIL;
  z.zfree := NIL;
  z.opaque := NIL;

  err := inflateInit ( z );
  IF err <> Z_OK
  THEN BEGIN
     UnCompressFileEx := - 3;
     Close ( fin );
     Close ( fout );
     Exit;
  END;

  aResult := 0;

  { check what type of file it is }
  BlockRead ( fin, four, sizeof  ( four ) ); { read first 4 bytes for signature }
  uSize := 0;

  IF ( strpas ( four ) = strpas ( htz_file_signature ) )
  THEN BEGIN { for htz files }
     Seek ( fin, 3 ); { read uncompressed size }
     BlockRead ( fin, HeadSize, Sizeof ( HeadSize ) );
     Seek ( fin, 9 ); { skip first 9 bytes }
     WITH HeadSize DO uSize := Size + Remainder;
  END ELSE FindDeflateStart ( fin );

  { get some memory }
  t1 := FileSize ( fin );
  IF uSize > 0 THEN t2 := uSize + 16 { uncompressed size + 16 bytes }
     ELSE t2 := GetOutBytes ( t1 );

  Getmem ( input_buffer, t1 );
  Getmem ( output_buffer, t2 );

   { loop: read from file in to buffer, decompress buffer,
          write buffer to file }
  WHILE ( err <> Z_STREAM_END )
  DO BEGIN
    { fill input buffer }
    blockread ( fin, input_buffer^, t1, in_count );

    { update the z_stream data structure }
    z.next_in := {addr} pointer ( input_buffer );  { where is the compressed data }
    z.avail_in := in_count;           { size of the compressed data }

    z.total_in := 0;        { allow us to monitor the input buffer usage }
    WHILE ( z.total_in < in_count ) DO  { until the input buffer is empty }
    BEGIN
      z.next_out := pointer ( output_buffer ); { where to put the uncom. data }
      z.avail_out := t2;   { max. space available }

      { now, decompress (inflate) }
      z.total_out := 0;    { allow us to know the output buffer size }
      err := inflate ( z, Z_NO_FLUSH );  { *** the inflate step **** }
      { inflate() stops when the input buffer becomes empty or the
        output buffer becomes full. }

      { flush out output buffer }
      blockwrite ( fout, output_buffer^, z.total_out );
      Inc ( aResult, z.total_out );
      IF ( Assigned ( CallBack ) ) AND ( z.total_out > 0 )
      THEN
        IF CallBack ( output_buffer^, z.total_out ) < 0
        THEN BEGIN
            err := Pred ( Z_OK );
        END;
      IF ( err = Z_STREAM_END ) OR ( err <> Z_OK ) THEN break;
    END;
    { end of decoding input buffer }

    IF ( err <> Z_OK ) THEN break;
  END;

  z.total_out := 0;
  err := inflateEnd ( z );

  { check error }
  IF err <> Z_OK
  THEN BEGIN
  END;

  { flush out output buffer }
  blockwrite ( fout, output_buffer^, z.total_out );
  Inc ( aResult, z.total_out );
  IF ( Assigned ( CallBack ) ) AND ( z.total_out > 0 )
     THEN IF CallBack ( output_buffer^, z.total_out ) < 0 THEN;

  { free some memory }
  freemem ( output_buffer, t2 );
  freemem ( input_buffer, t1 );

  close ( fin );
  close ( fout );

  { return }
  WITH HeadSize
  DO BEGIN
      t1 := Size + Remainder;
      IF ( t1 = aResult ) { then we have the correct uncompressed size }
        THEN UnCompressFileEx := aResult
          ELSE { error }
          { so return negative value : "- correct_size" * 10000 }
          UnCompressFileEx := (  - t1 ) * 10000;
  END;
END; { UnCompressFileEx }


{ @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ }
PROCEDURE GetModSize
( CONST Size : htz_Word; VAR Whole : htz_Word; VAR Remainder : Byte );
{
 Process a file size and break it down for inputting into
 the header
}
BEGIN
   Remainder := Size MOD 1024;
   Whole := Size - Remainder;
END; { GetModSize }


FUNCTION CompressFileEx ( Src, Target : pChar;
CallBack : htz_CallBack ) : Longint;
// STDCALL;
VAR
  Head : htz_header;
  err : int;    { error code }
  out_count,
  in_count : uLong;
  fin, fout : FILE;
  aResult : Longint;
  input_buffer,
  output_buffer : pzByteArray;
  t1, t2 : Longint;
  Junk2 : PACKED ARRAY [0..1] OF byte;

BEGIN
  CompressFileEx := - 1;

  { initialize input file }
  err := FileMode;
  FileMode := FileModeShare;
  assign ( fin, Strpas ( Src ) );
  reset ( fin, 1 );  { open }
  FileMode := err;
  IF IoResult <> 0 THEN Exit;

  { initialize output file }
  CompressFileEx := - 2;
  assign ( fout, Strpas ( Target ) );
  rewrite ( fout, 1 ); { create }
  IF IoResult <> 0
  THEN BEGIN
     Close ( fin );
     Exit;
  END;

  { get some memory }
  t1 := FileSize ( fin );    { save the size of the input file }
  t2 :=  Round ( t1 * 1.1 ); { allow some margin }

  Getmem ( input_buffer, t1 );
  Getmem ( output_buffer, t2 );

  { Prepare the header }
  WITH Head
  DO BEGIN
     signature [1] := $12;      { signature = first 3 bytes }
     signature [2] := $34;      {  "  }
     signature [3] := $00;      {  "  }
     GetModSize ( t1,
                Size_Main,      { 4-5 bytes = size 1  }
                Size_Rem );     { 6 bytes = size 2 (lo) }
     Last1 := $87;              { 7 bytes : don't know what to put
                                    here but many htz files have this }
     Last2 := $65;              { ditto }
     zlib_sig := z_deflate_sig; { zlib deflate marker }
  END;

  { write the header }
  BlockWrite ( fout, Head, Sizeof ( Head ) );

  { do the stuff }
  aResult := 0;
  WHILE NOT EOF ( fin )
  DO BEGIN
      out_count := t2;
      BlockRead ( fin, input_buffer^, t1, in_count );
      err := Compress{2} ( pBytef ( output_buffer ), out_count,
                        input_buffer^, in_count{, Z_BEST_COMPRESSION} );
      IF err = Z_OK
      THEN BEGIN
         BlockWrite ( fout, output_buffer^, out_count, out_count );
         Inc ( aResult, out_count );
         IF ( Assigned ( CallBack ) ) AND ( out_count > 0 )
            THEN IF CallBack ( output_buffer^, out_count ) < 0 THEN Break;
      END
      ELSE BEGIN
         aResult := err;
         Break;
      END;
  END; { while }

  { free some memory }
  Freemem ( output_buffer, t2 );
  Freemem ( input_buffer, t1 );

  { close files }
  Close ( fin );

  { write 2 extra bytes - our htz files are missing these bytes }
  Junk2 [0] := $DE;
  Junk2 [1] := $F3;
  BlockWrite ( fout, Junk2, Sizeof ( Junk2 ), out_count );
  IF ( Assigned ( CallBack ) ) AND ( out_count > 0 )
     THEN IF CallBack ( Junk2, out_count ) < 0 THEN;

  close ( fout );

  { return }
  CompressFileEx := aResult;
END; { CompressFileEx }

FUNCTION UnCompressFile ( Src, Target : pChar ) : Longint;
// STDCALL;
BEGIN
   UnCompressFile := UnCompressFileEx ( Src, Target, NIL );
END; { UnCompressFile }

FUNCTION CompressFile ( Src, Target : pChar ) : Longint;
// STDCALL;
BEGIN
   CompressFile := CompressFileEx ( Src, Target, NIL );
END; { CompressFile }

FUNCTION htz_Version : pChar;
// STDCALL;
BEGIN
   Result := Version_Num + #0;
END; { htz_Version }

FUNCTION htz_Author : pChar;
// STDCALL;
BEGIN
  Result := Author + #0;
END; { htz_Author }

FUNCTION htz_Date : pChar;
// STDCALL;
BEGIN
  Result := Version_Date + #0;
END; { htz_Date }

FUNCTION HTZFileSize ( FName : pChar ) : Longint;
// STDCALL;
VAR
  err : int;
  z : z_stream; { decompression stream }
  input_buffer,
  output_buffer : pzByteArray;
  t1, t2 : Longint;
  in_count : uLong;
  fin : FILE;
  aResult : Longint;
  uSize : Cardinal;
  four : sig_type;
  HeadSize : PACKED RECORD
     Size : htz_Word;
     Remainder : Byte;
  END;

BEGIN
  HTZFileSize := - 1;

  { initialize input file }
  err := FileMode;
  FileMode := FileModeShare;
  assign ( fin, Strpas ( FName ) );
  reset ( fin, 1 );  { open }
  FileMode := err;
  IF IoResult <> 0 THEN Exit;

  { initialize z_stream }
  z.zalloc := NIL;
  z.zfree := NIL;
  z.opaque := NIL;

  err := inflateInit ( z );
  IF err <> Z_OK
  THEN BEGIN
     HTZFileSize := - 3;
     Close ( fin );
     Exit;
  END;

  aResult := 0;

  { check what type of file it is }
  BlockRead ( fin, four, sizeof  ( four ) ); { read first 4 bytes for signature }
  uSize := 0;
  IF ( strpas ( four ) = strpas ( htz_file_signature ) )
  THEN BEGIN { for htz files }
     Seek ( fin, 3 ); { read uncompressed size }
     BlockRead ( fin, HeadSize, Sizeof ( HeadSize ) );
     Seek ( fin, 9 ); { skip first 9 bytes }
     WITH HeadSize DO Result := Size + Remainder;
     Close ( Fin );
     Exit;
  END
  ELSE BEGIN
     IF FindDeflateStart ( fin ) > 0
     THEN  { some compressed file }
     ELSE BEGIN
         Result := FileSize ( fin );
         Close ( fin );
         Exit;
     END;
  END;

  { get some memory }
  t1 := FileSize ( fin );
  IF uSize > 0 THEN t2 := uSize + 16 { uncompressed size + 16 bytes }
     ELSE t2 := GetOutBytes ( t1 );

  Getmem ( input_buffer, t1 );
  Getmem ( output_buffer, t2 );

  { loop: read from file in to buffer, decompress buffer, and return
     total uncompressed size }
  WHILE ( err <> Z_STREAM_END )
  DO BEGIN
    { fill input buffer }
    blockread ( fin, input_buffer^, t1, in_count );

    { update the z_stream data structure }
    z.next_in := {addr} pointer ( input_buffer );  { where is the compressed data }
    z.avail_in := in_count;           { size of the compressed data }

    z.total_in := 0;        { allow us to monitor the input buffer usage }
    WHILE ( z.total_in < in_count ) DO  { until the input buffer is empty }
    BEGIN
      z.next_out := pointer ( output_buffer ); { where to put the uncom. data }
      z.avail_out := t2;   { max. space available }

      { now, decompress (inflate) }
      z.total_out := 0;    { allow us to know the output buffer size }
      err := inflate ( z, Z_NO_FLUSH );  { *** the inflate step **** }
      { inflate() stops when the input buffer becomes empty or the
        output buffer becomes full. }

      Inc ( aResult, z.total_out );
      IF ( err = Z_STREAM_END ) OR ( err <> Z_OK ) THEN break;
    END;
    { end of decoding input buffer }

    IF ( err <> Z_OK ) THEN break;
  END;

  z.total_out := 0;
  inflateEnd ( z );

  { flush out output buffer }
  Inc ( aResult, z.total_out );

  { free some memory }
  freemem ( output_buffer, t2 );
  freemem ( input_buffer, t1 );

  close ( fin );
  HTZFileSize := aResult
END; { HTZFileSize }


{ @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ }
END.

